'use strict';

/* --------------------------------------------------------------
 cart_handler.js 2020-06-24
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Component for handling the add to cart and wishlist features
 * at the product details and the category listing pages. It cares
 * for attributes, properties, quantity and all other
 * relevant data for adding an item to the basket or wishlist
 */
gambio.widgets.module('cart_handler', ['hooks', 'form', 'xhr', 'loading_spinner', gambio.source + '/libs/events', gambio.source + '/libs/modal.ext-magnific', gambio.source + '/libs/modal'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        $body = $('body'),
        $window = $(window),
        busy = false,
        ajax = null,
        timeout = 0,
        defaults = {
        // AJAX "add to cart" URL
        addCartUrl: 'shop.php?do=Cart/BuyProduct',
        // AJAX "add to cart" URL for customizer products
        addCartCustomizerUrl: 'shop.php?do=Cart/Add',
        // AJAX URL to perform a value check
        checkUrl: 'shop.php?do=CheckStatus',
        // AJAX URL to perform the add to wishlist
        wishlistUrl: 'shop.php?do=WishList/Add',
        // Submit URL for price offer button
        priceOfferUrl: 'gm_price_offer.php',
        // Submit method for price offer
        priceOfferMethod: 'get',
        // Selector for the cart dropdown
        dropdown: '#head_shopping_cart',
        // "Add to cart" buttons selectors
        cartButtons: '.js-btn-add-to-cart',
        // "Wishlist" buttons selectors
        wishlistButtons: '.btn-wishlist',
        // "Price offer" buttons selectors
        priceOfferButtons: '.btn-price-offer',
        // Selector for the attribute fields
        attributes: '.js-calculate',
        // Selector for product property
        productOptions: '.modifier-group .modifier-content .modifier-item',
        productOptionField: '.hidden-input',
        // Selector for the quantity
        quantity: '.js-calculate-qty',
        // URL where to get the theme for the dropdown
        tpl: null,
        // Show attribute images in product images swiper (if possible)
        // -- this feature is not supported yet --
        attributImagesSwiper: false,
        // Trigger the attribute images to this selectors
        triggerAttrImagesTo: '#product_image_swiper, #product_thumbnail_swiper, ' + '#product_thumbnail_swiper_mobile',
        // Class that gets added to the button on processing
        processingClass: 'loading',
        // Duration for that the success or fail class gets added to the button
        processingDuration: 2000,
        // AJAX response content selectors
        selectorMapping: {
            buttons: '.shopping-cart-button',
            giftContent: '.gift-cart-content-wrapper',
            giftLayer: '.gift-cart-layer',
            shareContent: '.share-cart-content-wrapper',
            shareLayer: '.share-cart-layer',
            hiddenOptions: '#cart_quantity .hidden-options',
            message: '.global-error-messages',
            messageCart: '.cart-error-msg',
            messageHelp: '.help-block',
            modelNumber: '.model-number',
            modelNumberText: '.model-number-text',
            price: '.current-price-container',
            modifiersForm: '.modifiers-selection',
            quantity: '.products-quantity-value',
            ribbonSpecial: '.ribbon-special',
            shippingInformation: '#shipping-information-layer',
            shippingTime: '.products-shipping-time-value',
            shippingTimeImage: '.img-shipping-time img',
            totals: '#cart_quantity .total-box',
            weight: '.products-details-weight-container span'
        },
        page: 'product-listing'
    },
        options = $.extend(true, {}, defaults, data),
        module = {},
        mobile = $(window).width() <= 767;

    // ########## HELPER FUNCTIONS ##########

    /**
     * Helper function that updates the button
     * state with an error or success class for
     * a specified duration
     * @param   {object}        $target         jQuery selection of the target button
     * @param   {string}        state           The state string that gets added to the loading class
     * @private
     */
    var _addButtonState = function _addButtonState($target, state) {
        var timer = setTimeout(function () {
            $target.removeClass(options.processingClass + ' ' + options.processingClass + state);
        }, options.processingDuration);

        $target.data('timer', timer).addClass(options.processingClass + state);
    };

    /**
     * Helper function to set the messages and the
     * button state.
     * @param       {object}    data                Result form the ajax request
     * @param       {object}    $form               jQuery selecion of the form
     * @param       {boolean}   disableButtons      If true, the button state gets set to (in)active
     * @param       {boolean}   showNoCombiMesssage If true, the error message for missing property combination selection will be displayed
     * @private
     */
    var _stateManager = function _stateManager(data, $form, disableButtons, showNoCombiSelectedMesssage) {

        // Remove the attribute images from the common content
        // so that it doesn't get rendered anymore. Then trigger
        // an event to the given selectors and deliver the
        // attrImages object
        if (options.attributImagesSwiper && data.attrImages && data.attrImages.length) {
            delete data.content.images;
            $(options.triggerAttrImagesTo).trigger(jse.libs.theme.events.SLIDES_UPDATE(), { attributes: data.attrImages });
        }

        // Set the messages given inside the data.content object
        $.each(data.content, function (i, v) {
            var $element = $form.parent().find(options.selectorMapping[v.selector]);

            if ((!showNoCombiSelectedMesssage || v.value === '') && i === 'messageNoCombiSelected') {
                return true;
            }

            switch (v.type) {
                case 'hide':
                    if (v.value === 'true') {
                        $element.hide();
                    } else {
                        $element.show();
                    }
                    break;
                case 'html':
                    $element.html(v.value);
                    break;
                case 'attribute':
                    $element.attr(v.key, v.value);
                    break;
                case 'replace':
                    if (v.value) {
                        $element.replaceWith(v.value);
                    } else {
                        $element.addClass('hidden').empty();
                    }
                    break;
                default:
                    $element.text(v.value);
                    break;
            }
        });

        // Dis- / Enable the buttons
        if (disableButtons) {
            var $buttons = $form.find(options.cartButtons);
            if (data.success) {
                $buttons.removeClass('inactive');
                $buttons.removeClass('btn-inactive');
                $buttons.prop("disabled", false);
            } else {
                $buttons.addClass('inactive');
                $buttons.addClass('btn-inactive');
                $buttons.prop("disabled", true);
            }
        }

        if (data.content.message) {
            var $errorField = $form.find(options.selectorMapping[data.content.message.selector]);
            if (data.content.message.value) {
                $errorField.removeClass('hidden').show();
            } else {
                $errorField.addClass('hidden').hide();

                if (showNoCombiSelectedMesssage && data.content.messageNoCombiSelected !== undefined && data.content.messageNoCombiSelected) {
                    if (data.content.messageNoCombiSelected.value) {
                        $errorField.removeClass('hidden').show();
                    } else {
                        $errorField.addClass('hidden').hide();
                    }
                }
            }
        }

        $window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
    };

    /**
     * Helper function to send the ajax
     * On success redirect to a given url, open a layer with
     * a message or add the item to the cart-dropdown directly
     * (by triggering an event to the body)
     * @param       {object}      data      Form data
     * @param       {object}      $form     The form to fill
     * @param       {string}      url       The URL for the AJAX request
     * @private
     */
    var _addToSomewhere = function _addToSomewhere(data, $form, url, $button) {
        function callback() {
            jse.libs.xhr.post({ url: url, data: data }, true).done(function (result) {
                try {
                    // Fill the page with the result from the ajax
                    _stateManager(result, $form, false);

                    // If the AJAX was successful execute
                    // a custom functionality
                    if (result.success) {
                        switch (result.type) {
                            case 'url':
                                if (result.url.substr(0, 4) !== 'http') {
                                    location.href = jse.core.config.get('appUrl') + '/' + result.url;
                                } else {
                                    location.href = result.url;
                                }

                                break;
                            case 'dropdown':
                                $body.trigger(jse.libs.theme.events.CART_UPDATE(), [true]);
                                break;
                            case 'layer':
                                jse.libs.theme.modal.info({ title: result.title, content: result.msg });
                                break;
                            default:
                                break;
                        }
                    }
                } catch (ignore) {}
                _addButtonState($button, '-success');
            }).fail(function () {
                _addButtonState($button, '-fail');
            }).always(function () {
                // Reset the busy flag to be able to perform
                // further AJAX requests
                busy = false;
            });
        }

        if (!busy) {
            // only execute the ajax
            // if there is no pending ajax call
            busy = true;

            jse.libs.hooks.execute(jse.libs.hooks.keys.shop.cart.add, data, 500).then(callback).catch(callback);
        }
    };

    // ########## EVENT HANDLER ##########

    /**
     * Handler for the submit form / click
     * on "add to cart" & "wishlist" button.
     * It performs a check on the availability
     * of the combination and quantity. If
     * successful it performs the add to cart
     * or wishlist action, if it's not a
     * "check" call
     * @param       {object}    e      jQuery event object
     * @private
     */
    var _submitHandler = function _submitHandler(e) {
        if (e) {
            e.preventDefault();
        }

        var $self = $(this),
            $form = $self.is('form') ? $self : $self.closest('form'),
            customizer = $form.hasClass('customizer'),
            properties = !!$form.find('.properties-selection-form').length,
            module = properties ? '' : '/Attributes',
            showNoCombiSelectedMesssage = e && e.data && e.data.target && e.data.target !== 'check';

        if ($form.length) {

            // Show properties overlay
            // to disable user interaction
            // before markup replace
            if (properties) {
                $this.addClass('loading');
            }

            if ($self.is('select')) {
                var price = $self.find(":selected").attr('data-price');
                $self.parents('.modifier-group').find('.selected-value-price').text(price);
            }

            var getGalleryHash = $('#current-gallery-hash').val();
            $form.find('#update-gallery-hash').val(getGalleryHash);

            var formdata = jse.libs.form.getData($form, null, true);
            formdata.target = e && e.data && e.data.target ? e.data.target : 'check';
            formdata.isProductInfo = $form.hasClass('product-info') ? 1 : 0;

            // Abort previous check ajax if
            // there is one in progress
            if (ajax && e) {
                ajax.abort();
            }

            // Add processing-class to the button
            // and remove old timed events
            if (formdata.target !== 'check') {
                var timer = $self.data('timer');
                if (timer) {
                    clearTimeout(timer);
                }

                $self.removeClass(options.processingClass + '-success ' + options.processingClass + '-fail').addClass(options.processingClass);
            }

            ajax = jse.libs.xhr.get({
                url: options.checkUrl + module,
                data: formdata
            }, true).done(function (result) {
                _stateManager(result, $form, true, showNoCombiSelectedMesssage);
                $this.removeClass('loading');

                if (result.success) {
                    var event = null,
                        url = null;

                    switch (formdata.target) {
                        case 'wishlist':
                            if (customizer) {
                                event = jse.libs.theme.events.ADD_CUSTOMIZER_WISHLIST();
                            }
                            url = options.wishlistUrl;
                            break;
                        case 'cart':
                            if (customizer) {
                                event = jse.libs.theme.events.ADD_CUSTOMIZER_CART();
                                url = options.addCartCustomizerUrl;
                            } else {
                                url = options.addCartUrl;
                            }
                            break;
                        case 'price_offer':
                            $form.attr('action', options.priceOfferUrl).attr('method', options.priceOfferMethod);
                            $form.off('submit');
                            $form.submit();

                            return;
                        default:
                            setTimeout(function () {
                                $window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
                            }, 250);
                            break;
                    }

                    if (event) {
                        var deferred = $.Deferred();
                        deferred.done(function (customizerRandom) {
                            formdata[customizerRandom] = 0;
                            _addToSomewhere(formdata, $form, url, $self);
                        }).fail(function () {
                            _addButtonState($self, '-fail');
                        });
                        $body.trigger(event, [{ 'deferred': deferred, 'dataset': formdata }]);
                    } else if (url) {
                        _addToSomewhere(formdata, $form, url, $self);
                    }
                }

                // Check if the gallery images changed
                if (result.content.imageGallery.trim() !== '' && result.content.replaceGallery === true) {

                    var galleryLoading = jse.libs.loading_spinner.show($('.product-info-stage'), 9999);

                    var mainImageGallery = result.content.imageGallery;
                    var modalImageGallery = result.content.imageModal;

                    var mainGalleryWrapper = $('#image-collection-container');
                    var modalGalleryWrapper = $('#product_image_layer');

                    mainGalleryWrapper.html(mainImageGallery);
                    modalGalleryWrapper.html(modalImageGallery);

                    gambio.widgets.init($('#product_image_swiper')).done(function (arg1, arg2, arg3) {
                        var container = $('#product_image_swiper');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                        if (galleryLoading !== undefined) {
                            jse.libs.loading_spinner.hide(galleryLoading);
                        }

                        _initializeProductSwiperThumbnail($('#product_thumbnail_swiper'));
                    });

                    gambio.widgets.init($('#product_thumbnail_swiper_mobile')).done(function (arg1, arg2, arg3) {
                        var container = $('#product_thumbnail_swiper_mobile');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                        if (galleryLoading !== undefined) {
                            jse.libs.loading_spinner.hide(galleryLoading);
                        }

                        _initializeProductSwiperThumbnail($('#product_thumbnail_swiper_mobile'));
                    });
                    gambio.widgets.init($('.product-info-image')).done(function () {
                        var container = $('.product-info-image');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                    });
                    gambio.widgets.init($('#product-info-layer-image')).done(function () {
                        var container = $('#product-info-layer-image');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                    });
                    gambio.widgets.init($('#product-info-layer-thumbnails')).done(function () {
                        var container = $('#product-info-layer-thumbnails');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                    });
                }
            }).fail(function () {
                _addButtonState($self, '-fail');
            });
        }
    };

    /**
     * Handler for the change property option
     * */
    var _changeProductOptions = function _changeProductOptions(e) {
        var option = e.currentTarget;
        var optionValue = $(option).data('value');
        var optionContainer = $(option).parents('.modifier-group');

        $(optionContainer).find('li.active').removeClass('active');
        $(optionContainer).find('.modifier-item.active-modifier').removeClass('active-modifier');
        $(optionContainer).find('input.hidden-input').val(optionValue);
        $(optionContainer).find('input.hidden-input').trigger('blur', []);

        $(option).parents('li').addClass('active');
        $(option).addClass('active-modifier');
    };

    var _selectSelectedModifierInfo = function _selectSelectedModifierInfo(e) {
        var option = e.currentTarget;
        var price = $(option).attr('data-price');
        var label = $(option).attr('data-label');
        $(option).parents('.modifier-group').find('.selected-value-price').removeClass('temporary-value').attr('data-default-price', price);
        $(option).parents('.modifier-group').find('.selected-value').attr('data-default-value', label);
    };

    var _setSelectedModifierInfo = function _setSelectedModifierInfo(e) {
        var option = e.currentTarget;
        if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
            var price = $(option).attr('data-price');
            var label = $(option).attr('data-label');
            $(option).parents('.modifier-group').find('.selected-value-price').addClass('temporary-value').text(price);
            $(option).parents('.modifier-group').find('.selected-value').text(label);
        }
    };

    var _resetSelectedModifierInfo = function _resetSelectedModifierInfo(e) {
        var option = $(this);
        if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
            var priceHolder = $(option).parents('.modifier-group').find('.selected-value-price');
            var labelHolder = $(option).parents('.modifier-group').find('.selected-value');
            $(priceHolder).removeClass('temporary-value').text($(priceHolder).attr('data-default-price'));
            $(labelHolder).text($(labelHolder).attr('data-default-value'));
        }
    };

    /**
     * Keyup handler for quantity input field
     *
     * @param e
     * @private
     */
    var _keyupHandler = function _keyupHandler(e) {
        clearTimeout(timeout);

        timeout = setTimeout(function () {
            _submitHandler.call(this, e);
        }.bind(this), 300);
    };

    /**
     * Event handler for the add to cart button, that shows or hides the throbber.
     */
    var _addToCartThrobberHandler = function _addToCartThrobberHandler(e) {
        var $btn = $(this);
        var $btnFake = $this.find(".btn-add-to-cart-fake");
        var formReady = true;

        $(".properties-selection-form select").each(function () {
            var val = $(this).val();
            if (!val || val < 1) {
                formReady = false;
            }
        });

        if (formReady) {
            $btn.hide();
            $btnFake.show().prop("disabled", true).prepend('<span class="throbbler"></span>');
        }
    };

    /**
     * Cart dropdown oben event handler for the body.
     */
    var _cartDropdownOpenHandler = function _cartDropdownOpenHandler(e) {
        var $btn = $this.find("[name=btn-add-to-cart]");
        var $btnFake = $this.find(".btn-add-to-cart-fake");
        var fakeOrigLabel = $btnFake.html();
        var productCount = $(".cart-products-count").html();

        var textPhrases = JSON.parse($('#product-details-text-phrases').html());
        console.log(textPhrases['productsInCartSuffix']);

        $btnFake.html("<i class=\"fa fa-check\"></i> " + parseInt(productCount) + textPhrases['productsInCartSuffix']).prop("disabled", true).addClass("btn-buy-complete");

        setTimeout(function () {
            $btnFake.html(fakeOrigLabel).removeClass("btn-buy-complete").hide().prop("disabled", false);
            $(".throbbler", $btn).remove();
            $btn.show();
        }, 5000);
    };

    var _initializeProductSwiperThumbnail = function _initializeProductSwiperThumbnail($element) {
        gambio.widgets.init($element).done(function () {
            if ($element.swiper() !== undefined) {
                $element.swiper().update();
            }
            $element.trigger(jse.libs.theme.events.SLIDES_UPDATE());
        });
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget
     * @constructor
     */
    module.init = function (done) {

        var $forms = $this.find('form');

        if (options.page === 'product-info') {
            $forms.find("[name=btn-add-to-cart]").on('touchstart touchmove touchend touchcancel', function () {
                return $forms.find("[name=btn-add-to-cart]").click();
            });
            $forms.find("[name=btn-add-to-cart]").on('mouseup', _addToCartThrobberHandler);
            $("body").on('CART_DROPDOWN_OPEN', _cartDropdownOpenHandler);
        }

        $forms.on('submit', { 'target': 'cart' }, _submitHandler).on('click', options.wishlistButtons, { 'target': 'wishlist' }, _submitHandler).on('click', options.priceOfferButtons, { 'target': 'price_offer' }, _submitHandler).on('change', options.attributes, { 'target': 'check' }, _submitHandler).on('mouseover', options.attributes, _setSelectedModifierInfo).on('mouseout', options.attributes, _resetSelectedModifierInfo).on('blur', options.productOptionField, { 'target': 'check' }, _submitHandler).on('click', options.productOptions, { 'target': 'check' }, function (e) {
            _selectSelectedModifierInfo(e);
            _changeProductOptions(e);
        }).on('mouseover', options.productOptions, _setSelectedModifierInfo).on('mouseout', options.productOptions, _resetSelectedModifierInfo).on('blur', options.quantity, { 'target': 'check' }, function (e) {
            _submitHandler(e);
        }).on('keyup', options.quantity, { 'target': 'check' }, _keyupHandler);

        // Fallback if the backend renders incorrect data
        // on initial page call
        $forms.not('.no-status-check').each(function () {
            _submitHandler.call($(this));
        });

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
